function plot_params_CDS(CDX, param, subset)
% --------------------------------------------------------------------------------------------------
% Plot CDS-specific parameter time series
% --------------------------------------------------------------------------------------------------
% CDX       ... credit index structure
% param     ... which parameter to plot
% subset    ... optional list of tickers, that determine which CDS' parameters to plot
% --------------------------------------------------------------------------------------------------
% sample call: plot_params_CDS(cdx_tv_quarterly, 'a_i');
%              plot_params_CDS(cdx_tv_quarterly, 'a_i', {'AA'; 'AIG'; 'DIS'; 'MOT'});
% --------------------------------------------------------------------------------------------------

% Extract desired parameter time series
[a, theta_tilde, l_tilde, lambda] = get_portfolio_parameters(CDX, CDX.dates{1}(1), CDX.dates{1}(end));
num_firms = size(a, 2);
if strcmp(param, 'a_i')
    params = a;
elseif strcmp(param, 'theta_i')
    warning('Only plots idiosyncratic theta');
    params = theta_tilde;
elseif strcmp(param, 'sigma_i')
    params = a .* repmat(sqrt(CDX.AJD_common_factor.sigma), 1, num_firms);
elseif strcmp(param, 'L_i')
    warning('Only plots jump intensity');
    params = l_tilde;
elseif strcmp(param, 'mu_i')
    params = a .* repmat(CDX.AJD_common_factor.mu, 1, num_firms);
elseif strcmp(param, 'X_i')
    params = zeros(size(CDX.index_members));
    for i=1:length(CDX.dates{1})
        date = CDX.dates{1}(i);
        X_it = get_x_it(CDX, date, 2);
        params(i,:) = X_it';
    end
elseif strcmp(param, 'lambda_i')
    params = lambda;
end

% Replace missing values by NaNs
params(params < 0) = NaN;

% Select subset (if desired)
if (nargin == 3)
    all_tickers = {CDX.portfolio.ticker}';
    pos = [];
    for j=1:length(subset)
        pos2 = find(strcmp(all_tickers, subset{j}));
        if (~isempty(pos2))
            pos = [pos; pos2];
        end
    end
    params = params(:,pos);
end

% Make plot
dates = CDX.dates{1};
date_scale = dates - dates(1);
plot(date_scale, params, 'LineWidth', 1.2)
if (nargin == 3)
    if (length(pos) < 20)
        legend(subset);
    end
end

% Make nice axes & labels
special = strfind(param, '_');
for i=length(special):(-1):1
    param = [param(1:(special(i)-1)) param((special(i)+1):end)];
end
ylabel(param);
make_nice_axis(gca, reshape(params, [numel(params), 1]));
make_nice_ticks(dates, 122);

